<?php
ob_start(); // Start output buffering
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/database.php';

$database = new Database();
$db = $database->getConnection();
$auth = new Auth($database);

// Check permissions
if (!$auth->hasPermission('president')) {
    header('Location: index.php');
    exit();
}

$error = null;

// Handle settings operations
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    error_log("POST request received: " . print_r($_POST, true));
    
    if (isset($_POST['action'])) {
        error_log("Action: " . $_POST['action']);
        
        switch ($_POST['action']) {
            case 'update_association':
                // Handle logo upload
                $logo_path = isset($settings['logo_path']) ? $settings['logo_path'] : null;
                
                if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                    $allowed_types = ['image/jpeg', 'image/png'];
                    $max_size = 2 * 1024 * 1024; // 2MB
                    
                    // Debug information
                    error_log("File upload debug:");
                    error_log("File type: " . $_FILES['logo']['type']);
                    error_log("File size: " . $_FILES['logo']['size']);
                    error_log("Temp name: " . $_FILES['logo']['tmp_name']);
                    
                    if (!in_array($_FILES['logo']['type'], $allowed_types)) {
                        $error = 'يجب أن يكون الملف بصيغة JPG أو PNG';
                        error_log("File type not allowed: " . $_FILES['logo']['type']);
                        break;
                    }
                    
                    if ($_FILES['logo']['size'] > $max_size) {
                        $error = 'حجم الملف يجب أن لا يتجاوز 2 ميغابايت';
                        error_log("File too large: " . $_FILES['logo']['size']);
                        break;
                    }
                    
                    // Create uploads directory if it doesn't exist
                    $upload_dir = __DIR__ . '/../uploads';
                    if (!file_exists($upload_dir)) {
                        if (!mkdir($upload_dir, 0777, true)) {
                            $error = 'فشل في إنشاء مجلد التحميل';
                            error_log("Failed to create upload directory: " . $upload_dir);
                            break;
                        }
                    }

                    // Delete all existing files in uploads directory
                    $files = glob($upload_dir . '/*');
                    foreach($files as $file) {
                        if(is_file($file) && !in_array(basename($file), ['.htaccess'])) {
                            unlink($file);
                        }
                    }
                    
                    // Generate unique filename
                    $file_ext = strtolower(pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION));
                    $filename = 'association_logo.' . $file_ext;
                    $upload_path = $upload_dir . '/' . $filename;
                    
                    error_log("Upload path: " . $upload_path);
                    
                    // Delete old logo if exists
                    if ($logo_path && file_exists(__DIR__ . '/../' . $logo_path)) {
                        if (!unlink(__DIR__ . '/../' . $logo_path)) {
                            error_log("Failed to delete old logo: " . __DIR__ . '/../' . $logo_path);
                        }
                    }
                    
                    // Move uploaded file
                    if (move_uploaded_file($_FILES['logo']['tmp_name'], $upload_path)) {
                        $logo_path = 'uploads/' . $filename;
                        error_log("File uploaded successfully to: " . $upload_path);
                        
                        // Set proper permissions
                        chmod($upload_path, 0644);
                    } else {
                        $error = 'حدث خطأ أثناء رفع الملف';
                        error_log("Failed to move uploaded file from " . $_FILES['logo']['tmp_name'] . " to " . $upload_path);
                        error_log("Upload error: " . error_get_last()['message']);
                        break;
                    }
                }

                try {
                    error_log("Saving to database with logo_path: " . $logo_path);
                    $stmt = $db->prepare("INSERT OR REPLACE INTO association_settings (id, name, address, phone, email, logo_path) VALUES (1, ?, ?, ?, ?, ?)");
                    $stmt->execute([
                        $_POST['name'],
                        $_POST['address'],
                        $_POST['phone'],
                        $_POST['email'],
                        $logo_path
                    ]);
                    $_SESSION['success'] = 'تم حفظ المعلومات بنجاح';
                    header('Location: index.php?page=settings');
                    exit();
                } catch (PDOException $e) {
                    $error = 'حدث خطأ أثناء حفظ المعلومات: ' . $e->getMessage();
                    error_log("Database error: " . $e->getMessage());
                }
                break;

            case 'add_bank_account':
                $stmt = $db->prepare("INSERT INTO bank_accounts (bank_name, account_number, description) VALUES (?, ?, ?)");
                $stmt->execute([
                    $_POST['bank_name'],
                    $_POST['account_number'],
                    $_POST['description']
                ]);
                break;

            case 'delete_bank_account':
                $stmt = $db->prepare("DELETE FROM bank_accounts WHERE id = ?");
                $stmt->execute([$_POST['account_id']]);
                break;

            case 'add_asset':
                $stmt = $db->prepare("INSERT INTO assets (name, description, status, acquisition_date) VALUES (?, ?, ?, ?)");
                $stmt->execute([
                    $_POST['name'],
                    $_POST['description'],
                    $_POST['status'],
                    $_POST['acquisition_date']
                ]);
                break;

            case 'update_asset':
                $stmt = $db->prepare("UPDATE assets SET name = ?, description = ?, status = ? WHERE id = ?");
                $stmt->execute([
                    $_POST['name'],
                    $_POST['description'],
                    $_POST['status'],
                    $_POST['asset_id']
                ]);
                break;

            case 'delete_asset':
                $stmt = $db->prepare("DELETE FROM assets WHERE id = ?");
                $stmt->execute([$_POST['asset_id']]);
                break;
        }
        header('Location: index.php?page=settings');
        exit();
    }
}

// Get association settings
$stmt = $db->query("SELECT * FROM association_settings WHERE id = 1");
$settings = $stmt->fetch(PDO::FETCH_ASSOC);

// Ensure uploads directory exists
$upload_dir = __DIR__ . '/../uploads';
if (!file_exists($upload_dir)) {
    if (!mkdir($upload_dir, 0777, true)) {
        error_log("Failed to create upload directory: " . $upload_dir);
    } else {
        error_log("Created upload directory: " . $upload_dir);
    }
}

// Debug information
error_log("Settings from database: " . print_r($settings, true));

// Define base URL for images
$base_url = '/mdm/';

// Get bank accounts
$stmt = $db->query("SELECT * FROM bank_accounts ORDER BY bank_name");
$bankAccounts = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get assets
$stmt = $db->query("SELECT * FROM assets ORDER BY name");
$assets = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<div class="container-fluid">
    <?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?php echo htmlspecialchars($error); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['success'])): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?php echo htmlspecialchars($_SESSION['success']); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    <?php unset($_SESSION['success']); ?>
    <?php endif; ?>

    <!-- Debug information -->
    <?php if (isset($_FILES['logo'])): ?>
    <div class="alert alert-info">
        <h6>معلومات التصحيح:</h6>
        <pre><?php print_r($_FILES['logo']); ?></pre>
    </div>
    <?php endif; ?>

    <!-- Association Settings -->
    <div class="row mb-4">
        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h5 class="card-title mb-0">معلومات الجمعية</h5>
                </div>
                <div class="card-body">
                    <form method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="update_association">
                        <div class="mb-3">
                            <label class="form-label">شعار الجمعية</label>
                            <img src="uploads/association_logo.png"  style="height: 40px;">
                            <?php if (!empty($settings['logo_path']) && file_exists(__DIR__ . '/../' . $settings['logo_path'])): ?>
                            <div class="mb-2">
                                <img src="<?php echo $base_url . htmlspecialchars($settings['logo_path']); ?>" alt="شعار الجمعية" style="max-height: 100px;" class="mb-2">
                                <p class="text-muted small">Current logo path: <?php echo htmlspecialchars($settings['logo_path']); ?></p>
                            </div>
                            <?php else: ?>
                            <p class="text-muted small">No logo uploaded yet</p>
                            <?php endif; ?>
                            <input type="file" class="form-control" name="logo" accept="image/jpeg,image/png">
                            <small class="text-muted">يجب أن يكون الملف بصيغة JPG أو PNG وحجم لا يتجاوز 2 ميغابايت</small>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">اسم الجمعية</label>
                            <input type="text" class="form-control" name="name" value="<?php echo htmlspecialchars($settings['name'] ?? ''); ?>" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">العنوان</label>
                            <textarea class="form-control" name="address" rows="2"><?php echo htmlspecialchars($settings['address'] ?? ''); ?></textarea>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">الهاتف</label>
                            <input type="tel" class="form-control" name="phone" value="<?php echo htmlspecialchars($settings['phone'] ?? ''); ?>">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">البريد الإلكتروني</label>
                            <input type="email" class="form-control" name="email" value="<?php echo htmlspecialchars($settings['email'] ?? ''); ?>">
                        </div>
                        <button type="submit" class="btn btn-primary">حفظ التغييرات</button>
                    </form>
                </div>
            </div>
        </div>

        <!-- Bank Accounts -->
        <div class="col-md-6">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">الحسابات البنكية</h5>
                    <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#addBankAccountModal">
                        <i class="bi bi-plus-circle"></i> إضافة حساب
                    </button>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>البنك</th>
                                    <th>رقم الحساب</th>
                                    <th>الوصف</th>
                                    <th>العمليات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($bankAccounts as $account): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($account['bank_name']); ?></td>
                                    <td><?php echo htmlspecialchars($account['account_number']); ?></td>
                                    <td><?php echo htmlspecialchars($account['description']); ?></td>
                                    <td>
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="action" value="delete_bank_account">
                                            <input type="hidden" name="account_id" value="<?php echo $account['id']; ?>">
                                            <button type="submit" class="btn btn-sm btn-danger"
                                                    onclick="return confirm('هل أنت متأكد من حذف الحساب <?php echo htmlspecialchars($account['bank_name']); ?>؟');">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Assets Management -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">ممتلكات الجمعية</h5>
                    <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#addAssetModal">
                        <i class="bi bi-plus-circle"></i> إضافة ممتلكات
                    </button>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table datatable">
                            <thead>
                                <tr>
                                    <th>اسم الممتلك</th>
                                    <th>الوصف</th>
                                    <th>الحالة</th>
                                    <th>تاريخ الاقتناء</th>
                                    <th>العمليات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($assets as $asset): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($asset['name']); ?></td>
                                    <td><?php echo htmlspecialchars($asset['description']); ?></td>
                                    <td>
                                        <span class="badge <?php echo $asset['status'] === 'good' ? 'bg-success' : 'bg-warning'; ?>">
                                            <?php echo $asset['status'] === 'good' ? 'جيدة' : 'تحتاج صيانة'; ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($asset['acquisition_date']); ?></td>
                                    <td>
                                        <button class="btn btn-sm btn-primary edit-asset" 
                                                data-asset='<?php echo json_encode($asset); ?>'
                                                data-bs-toggle="modal" 
                                                data-bs-target="#editAssetModal">
                                            <i class="bi bi-pencil"></i>
                                        </button>
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="action" value="delete_asset">
                                            <input type="hidden" name="asset_id" value="<?php echo $asset['id']; ?>">
                                            <button type="submit" class="btn btn-sm btn-danger"
                                                    onclick="return confirm('هل أنت متأكد من حذف الممتلك <?php echo htmlspecialchars($asset['name']); ?>؟');">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Bank Account Modal -->
<div class="modal fade" id="addBankAccountModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">إضافة حساب بنكي</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add_bank_account">
                    <div class="mb-3">
                        <label class="form-label">اسم البنك</label>
                        <input type="text" class="form-control" name="bank_name" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">رقم الحساب</label>
                        <input type="text" class="form-control" name="account_number" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">الوصف</label>
                        <textarea class="form-control" name="description" rows="2"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-primary">حفظ</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Add Asset Modal -->
<div class="modal fade" id="addAssetModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">إضافة ممتلك</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add_asset">
                    <div class="mb-3">
                        <label class="form-label">اسم الممتلك</label>
                        <input type="text" class="form-control" name="name" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">الوصف</label>
                        <textarea class="form-control" name="description" rows="2"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">الحالة</label>
                        <select class="form-control" name="status" required>
                            <option value="good">جيدة</option>
                            <option value="maintenance">تحتاج صيانة</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">تاريخ الاقتناء</label>
                        <input type="date" class="form-control" name="acquisition_date" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-primary">حفظ</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Asset Modal -->
<div class="modal fade" id="editAssetModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">تعديل ممتلك</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="update_asset">
                    <input type="hidden" name="asset_id" id="edit_asset_id">
                    <div class="mb-3">
                        <label class="form-label">اسم الممتلك</label>
                        <input type="text" class="form-control" name="name" id="edit_asset_name" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">الوصف</label>
                        <textarea class="form-control" name="description" id="edit_asset_description" rows="2"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">الحالة</label>
                        <select class="form-control" name="status" id="edit_asset_status" required>
                            <option value="good">جيدة</option>
                            <option value="maintenance">تحتاج صيانة</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-primary">حفظ التغييرات</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
$(document).ready(function() {
    // Initialize edit asset modal
    $('.edit-asset').click(function() {
        const asset = $(this).data('asset');
        $('#edit_asset_id').val(asset.id);
        $('#edit_asset_name').val(asset.name);
        $('#edit_asset_description').val(asset.description);
        $('#edit_asset_status').val(asset.status);
    });
});
</script>
