<?php
// Fonctions utilitaires pour l'application

/**
 * Copie récursive d'un dossier et de son contenu
 * 
 * @param string $source Chemin du dossier source
 * @param string $destination Chemin du dossier de destination
 * @return bool Succès ou échec de la copie
 */
function copyDirectory($source, $destination) {
    // Créer le dossier de destination s'il n'existe pas
    if (!is_dir($destination)) {
        mkdir($destination, 0755, true);
    }
    
    // Ouvrir le dossier source
    $dir = opendir($source);
    
    // Parcourir tous les fichiers et dossiers
    while (($file = readdir($dir)) !== false) {
        if ($file != '.' && $file != '..') {
            $srcFile = $source . '/' . $file;
            $destFile = $destination . '/' . $file;
            
            if (is_dir($srcFile)) {
                // Si c'est un dossier, appel récursif
                copyDirectory($srcFile, $destFile);
            } else {
                // Si c'est un fichier, le copier
                copy($srcFile, $destFile);
            }
        }
    }
    
    // Fermer le dossier
    closedir($dir);
    
    return true;
}

/**
 * Génère un nom de dossier unique basé sur le nom du gérant
 * 
 * @param string $name Nom du gérant
 * @return string Nom du dossier
 */
function generateFolderName($name) {
    // Convertir en minuscules et remplacer les espaces par des tirets
    $folderName = strtolower(trim($name));
    $folderName = preg_replace('/[^a-z0-9]/', '-', $folderName);
    $folderName = preg_replace('/-+/', '-', $folderName);
    
    // Ajouter un timestamp pour garantir l'unicité
    $folderName .= '-' . time();
    
    return $folderName;
}

/**
 * Vérifie si un utilisateur est connecté en tant qu'admin
 * 
 * @return bool True si l'utilisateur est connecté, false sinon
 */
function isAdminLoggedIn() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

/**
 * Redirection avec un message flash
 * 
 * @param string $url URL de redirection
 * @param string $message Message à afficher
 * @param string $type Type de message (success, error, warning, info)
 */
function redirectWithMessage($url, $message, $type = 'info') {
    $_SESSION['flash_message'] = [
        'message' => $message,
        'type' => $type
    ];
    
    header("Location: $url");
    exit;
}

/**
 * Affiche un message flash s'il existe et le supprime
 * 
 * @return string|null HTML du message ou null s'il n'y a pas de message
 */
function displayFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message']['message'];
        $type = $_SESSION['flash_message']['type'];
        
        unset($_SESSION['flash_message']);
        
        return '<div class="alert alert-' . $type . '">' . $message . '</div>';
    }
    
    return null;
}
?>
