<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/print_header.php';

$database = new Database();
$db = $database->getConnection();
$auth = new Auth($database);

// Check permissions
if (!$auth->hasPermission('treasurer')) {
    header('Location: index.php');
    exit();
}

// Get current user ID
$currentUserId = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;

// Handle cash operations
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        try {
            if (!$currentUserId) {
                throw new Exception("يجب عليك تسجيل الدخول لإجراء هذه العملية");
            }

            switch ($_POST['action']) {
                case 'add_fund':
                    $stmt = $db->prepare("INSERT INTO cash_funds (name, description) VALUES (?, ?)");
                    $stmt->execute([
                        $_POST['name'],
                        $_POST['description']
                    ]);
                    break;

                case 'delete_fund':
                    // Delete all related transactions first
                    $db->beginTransaction();
                    try {
                        // Delete from financial_transactions
                        $stmt = $db->prepare("DELETE FROM financial_transactions WHERE (account_type = 'cash' AND account_id = ?) OR (transfer_to_type = 'cash' AND transfer_to_id = ?)");
                        $stmt->execute([$_POST['fund_id'], $_POST['fund_id']]);
                        
                        // Delete from cash_transactions
                        $stmt = $db->prepare("DELETE FROM cash_transactions WHERE fund_id = ?");
                        $stmt->execute([$_POST['fund_id']]);
                        
                        // Delete the fund
                        $stmt = $db->prepare("DELETE FROM cash_funds WHERE id = ?");
                        $stmt->execute([$_POST['fund_id']]);
                        
                        $db->commit();
                        $_SESSION['success'] = "تم حذف الصندوق وجميع عملياته بنجاح";
                        header("Location: index.php?page=cash_funds");
                        exit;
                    } catch (Exception $e) {
                        $db->rollBack();
                        throw $e;
                    }
                    break;

                case 'add_transaction':
                    $db->beginTransaction();
                    
                    // Ajouter la transaction
                    $stmt = $db->prepare("INSERT INTO cash_transactions (fund_id, type, amount, description, reference_number, transaction_date, created_by) 
                                        VALUES (?, ?, ?, ?, ?, ?, ?)");
                    $stmt->execute([
                        $_POST['fund_id'],
                        $_POST['type'],
                        $_POST['amount'],
                        $_POST['description'],
                        $_POST['reference_number'],
                        $_POST['transaction_date'],
                        $currentUserId
                    ]);

                    // Mettre à jour le solde de la caisse
                    $amount = $_POST['type'] === 'deposit' ? $_POST['amount'] : -$_POST['amount'];
                    $stmt = $db->prepare("UPDATE cash_funds SET balance = balance + ? WHERE id = ?");
                    $stmt->execute([$amount, $_POST['fund_id']]);

                    $db->commit();
                    break;

                case 'delete_transaction':
                    $db->beginTransaction();

                    // Récupérer les informations de la transaction
                    $stmt = $db->prepare("SELECT type, amount, fund_id FROM cash_transactions WHERE id = ?");
                    $stmt->execute([$_POST['transaction_id']]);
                    $transaction = $stmt->fetch(PDO::FETCH_ASSOC);

                    if ($transaction) {
                        // Annuler l'effet de la transaction sur le solde
                        $amount = $transaction['type'] === 'deposit' ? -$transaction['amount'] : $transaction['amount'];
                        $stmt = $db->prepare("UPDATE cash_funds SET balance = balance + ? WHERE id = ?");
                        $stmt->execute([$amount, $transaction['fund_id']]);

                        // Supprimer la transaction
                        $stmt = $db->prepare("DELETE FROM cash_transactions WHERE id = ?");
                        $stmt->execute([$_POST['transaction_id']]);
                    }

                    $db->commit();
                    break;
            }
            header('Location: index.php?page=cash_funds&success=1');
            exit();
        } catch (Exception $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            $error = $e->getMessage();
        }
    }
}

// Handle delete fund action
if (isset($_POST['action']) && $_POST['action'] === 'delete_fund' && isset($_POST['fund_id'])) {
    try {
        $db->beginTransaction();
        
        // Delete all related transactions first
        $stmt = $db->prepare("DELETE FROM financial_transactions WHERE (account_type = 'cash' AND account_id = ?) OR (transfer_to_type = 'cash' AND transfer_to_id = ?)");
        $stmt->execute([$_POST['fund_id'], $_POST['fund_id']]);
        
        // Delete from cash_transactions
        $stmt = $db->prepare("DELETE FROM cash_transactions WHERE fund_id = ?");
        $stmt->execute([$_POST['fund_id']]);
        
        // Delete the fund
        $stmt = $db->prepare("DELETE FROM cash_funds WHERE id = ?");
        $stmt->execute([$_POST['fund_id']]);
        
        $db->commit();
        echo json_encode(['success' => true, 'message' => 'تم حذف الصندوق وجميع عملياته بنجاح']);
        exit;
    } catch (Exception $e) {
        $db->rollBack();
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        exit;
    }
}

// Handle delete transaction action
if (isset($_POST['action']) && $_POST['action'] === 'delete_transaction' && isset($_POST['transaction_id'])) {
    try {
        $db->beginTransaction();

        // Get transaction details
        $stmt = $db->prepare("SELECT * FROM financial_transactions WHERE id = ? AND account_type = 'cash'");
        $stmt->execute([$_POST['transaction_id']]);
        $transaction = $stmt->fetch();

        if (!$transaction) {
            throw new Exception('العملية غير موجودة');
        }

        // Delete the transaction
        $stmt = $db->prepare("DELETE FROM financial_transactions WHERE id = ?");
        $stmt->execute([$_POST['transaction_id']]);

        $db->commit();
        echo json_encode(['success' => true, 'message' => 'تم حذف العملية بنجاح']);
        exit;
    } catch (Exception $e) {
        $db->rollBack();
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        exit;
    }
}

// Get cash funds with their balances
$stmt = $db->query("
    SELECT 
        cf.*,
        COALESCE(
            (
                -- Add incomes
                SELECT COALESCE(SUM(amount), 0)
                FROM financial_transactions
                WHERE account_type = 'cash' 
                AND account_id = cf.id 
                AND type = 'income'
            ) -
            -- Subtract expenses
            (
                SELECT COALESCE(SUM(amount), 0)
                FROM financial_transactions
                WHERE account_type = 'cash' 
                AND account_id = cf.id 
                AND type = 'expense'
            ) -
            -- Subtract outgoing transfers
            (
                SELECT COALESCE(SUM(amount), 0)
                FROM financial_transactions
                WHERE account_type = 'cash' 
                AND account_id = cf.id 
                AND type = 'transfer'
                AND transfer_to_type IS NOT NULL
            ) +
            -- Add incoming transfers
            (
                SELECT COALESCE(SUM(amount), 0)
                FROM financial_transactions
                WHERE transfer_to_type = 'cash' 
                AND transfer_to_id = cf.id 
                AND type = 'transfer'
            ), 0
        ) as current_balance,
        (
            SELECT COUNT(*)
            FROM financial_transactions
            WHERE (account_type = 'cash' AND account_id = cf.id)
            OR (transfer_to_type = 'cash' AND transfer_to_id = cf.id)
        ) as transaction_count
    FROM cash_funds cf
    ORDER BY cf.name
");
$cashFunds = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate total balance
$totalBalance = array_sum(array_column($cashFunds, 'current_balance'));

// Get cash funds for filter
$cashFundsStmt = $db->prepare("SELECT id, name FROM cash_funds ORDER BY name ASC");
$cashFundsStmt->execute();
$cashFundsFilter = $cashFundsStmt->fetchAll();

// Get filter parameters
$selectedFund = $_GET['fund_id'] ?? 'all';
$startDate = $_GET['start_date'] ?? '';
$endDate = $_GET['end_date'] ?? '';

// Get transactions based on filters
$query = "
    SELECT 
        ft.*,
        cf.name as fund_name,
        cf_to.name as transfer_to_name,
        u.username as created_by_name,
        CASE 
            WHEN ft.type = 'income' THEN ic.name
            WHEN ft.type = 'expense' THEN ec.name
            ELSE NULL
        END as category_name
    FROM financial_transactions ft
    LEFT JOIN cash_funds cf ON ft.account_type = 'cash' AND ft.account_id = cf.id
    LEFT JOIN cash_funds cf_to ON ft.transfer_to_type = 'cash' AND ft.transfer_to_id = cf_to.id
    LEFT JOIN income_categories ic ON ft.type = 'income' AND ft.category_id = ic.id
    LEFT JOIN expense_categories ec ON ft.type = 'expense' AND ft.category_id = ec.id
    LEFT JOIN users u ON ft.created_by = u.id
    WHERE 1=1 ";

if ($selectedFund !== 'all') {
    $query .= " AND (
        (ft.account_type = 'cash' AND ft.account_id = :fund_id) OR 
        (ft.transfer_to_type = 'cash' AND ft.transfer_to_id = :fund_id)
    )";
}

if ($startDate) {
    $query .= " AND DATE(ft.transaction_date) >= :start_date";
}

if ($endDate) {
    $query .= " AND DATE(ft.transaction_date) <= :end_date";
}

$query .= " ORDER BY ft.transaction_date DESC, ft.id DESC";

$stmt = $db->prepare($query);

if ($selectedFund !== 'all') {
    $stmt->bindParam(':fund_id', $selectedFund);
}

if ($startDate) {
    $stmt->bindParam(':start_date', $startDate);
}

if ($endDate) {
    $stmt->bindParam(':end_date', $endDate);
}

$stmt->execute();
$transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate totals for the filtered transactions
$totalsQuery = "
    SELECT 
        SUM(CASE WHEN type = 'income' THEN amount ELSE 0 END) as total_income,
        SUM(CASE WHEN type = 'expense' THEN amount ELSE 0 END) as total_expense,
        SUM(CASE 
            WHEN type = 'transfer' AND transfer_to_type IS NOT NULL THEN amount
            ELSE 0 
        END) as total_transfers_out,
        SUM(CASE 
            WHEN type = 'transfer' AND transfer_to_type IS NULL THEN amount
            ELSE 0 
        END) as total_transfers_in
    FROM financial_transactions ft
    WHERE 1=1 ";

if ($selectedFund !== 'all') {
    $totalsQuery .= " AND (
        (ft.account_type = 'cash' AND ft.account_id = :fund_id) OR 
        (ft.transfer_to_type = 'cash' AND ft.transfer_to_id = :fund_id)
    )";
}

if ($startDate) {
    $totalsQuery .= " AND DATE(ft.transaction_date) >= :start_date";
}

if ($endDate) {
    $totalsQuery .= " AND DATE(ft.transaction_date) <= :end_date";
}

$stmt = $db->prepare($totalsQuery);

if ($selectedFund !== 'all') {
    $stmt->bindParam(':fund_id', $selectedFund);
}

if ($startDate) {
    $stmt->bindParam(':start_date', $startDate);
}

if ($endDate) {
    $stmt->bindParam(':end_date', $endDate);
}

$stmt->execute();
$totals = $stmt->fetch(PDO::FETCH_ASSOC);

?>

<div class="container-fluid">
    <?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?php echo $error; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <!-- Cash Funds Summary -->
    <div class="row mb-4">
        <div class="col-md-6">
            <div class="card bg-primary text-white">
                <div class="card-body">
                    <h5 class="card-title">الرصيد الإجمالي للصناديق الفرعية- بالإضافة إلى المبالغ غير الموجهة</h5>
                    <h3 class="card-text"><?php echo number_format($totalBalance, 2); ?> درهم</h3>
                    <small>مجموع أرصدة <?php echo count($cashFunds)-1; ?> صندوق</small>
                </div>
            </div>
        </div>
        <div class="col-md-6">
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addFundModal">
                <i class="fas fa-plus"></i> إضافة صندوق جديد
            </button>
        </div>
    </div>

    <!-- Individual Cash Fund Cards -->
    <div class="row mb-4">
        <?php foreach ($cashFunds as $fund): ?>
        <div class="col-md-4 mb-3">
            <div class="card h-100">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start mb-2">
                        <h5 class="card-title"><?php echo htmlspecialchars($fund['name']); ?></h5>
                        <button class="btn btn-sm btn-outline-danger delete-fund" 
                                data-id="<?php echo $fund['id']; ?>"
                                data-name="<?php echo htmlspecialchars($fund['name']); ?>">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                    <h2 class="card-text">
                        <span class="fs-4 <?php echo $fund['current_balance'] >= 0 ? 'text-success' : 'text-danger'; ?>">
                            <?php echo number_format($fund['current_balance'], 2); ?> درهم
                        </span>
        </h2>
                  
                </div>
            </div>
        </div>
        <?php endforeach; ?>
    </div>

    <!-- Action Buttons -->
    <div class="row mb-4">
   
    </div>

    <!-- Filter Form -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="get" class="row g-3 align-items-end">
                <input type="hidden" name="page" value="cash_funds">
                
                <div class="col-md-3">
                    <label for="fund_id" class="form-label">الصندوق</label>
                    <select name="fund_id" id="fund_id" class="form-select">
                        <option value="all" <?php echo $selectedFund === 'all' ? 'selected' : ''; ?>>جميع الصناديق</option>
                        <?php foreach ($cashFunds as $fund): ?>
                            <option value="<?php echo $fund['id']; ?>" <?php echo $selectedFund == $fund['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($fund['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="col-md-3">
                    <label for="start_date" class="form-label">من تاريخ</label>
                    <input type="date" class="form-control" id="start_date" name="start_date" value="<?php echo $startDate; ?>">
                </div>

                <div class="col-md-3">
                    <label for="end_date" class="form-label">إلى تاريخ</label>
                    <input type="date" class="form-control" id="end_date" name="end_date" value="<?php echo $endDate; ?>">
                </div>

                <div class="col-md-3">
                    <button type="submit" class="btn btn-primary">تصفية</button>
                    <a href="?page=cash_funds" class="btn btn-secondary">إعادة تعيين</a>
                </div>
            </form>
        </div>
    </div>

    <!-- Transactions Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">سجل العمليات</h5>
            <button onclick="window.print()" class="btn btn-secondary btn-sm">
                <i class="fas fa-print"></i> طباعة
            </button>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <!-- Print Header -->
                <div class="d-none d-print-block mb-4">
                    <?php echo get_print_header('سجل عمليات الصندوق'); ?>
                </div>
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>التاريخ</th>
                            <th>النوع</th>
                            <th>المبلغ</th>
                            <th>تفاصيل التحويل</th>
                            <th class="actions-column">العمليات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($transactions as $transaction): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($transaction['transaction_date']); ?></td>
                            <td>
                                <?php 
                                switch($transaction['type']) {
                                    case 'income':
                                        echo '<span class="badge bg-success">مدخول</span>';
                                        break;
                                    case 'expense':
                                        echo '<span class="badge bg-danger">مصروف</span>';
                                        break;
                                    case 'transfer':
                                        echo '<span class="badge bg-warning text-dark">تحويل صادر</span>';
                                        break;
                                }
                                ?>
                            </td>
                            <td class="<?php echo $transaction['type'] === 'income' ? 'text-success' : 'text-danger'; ?>">
                                <?php
                                $amount = number_format($transaction['amount'], 2);
                                echo $transaction['type'] === 'income' ? "+ {$amount} درهم" : "- {$amount} درهم";
                                ?>
                            </td>
                            <td>
                                <?php 
                                if ($transaction['type'] === 'transfer') {
                                    echo htmlspecialchars($transaction['transfer_to_name'] ?? '');
                                } else {
                                    echo htmlspecialchars($transaction['description']);
                                }
                                ?>
                            </td>
                            <td class="actions-column">
                                <button type="button" class="btn btn-sm btn-danger delete-transaction" data-id="<?php echo $transaction['id']; ?>">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php if (empty($transactions)): ?>
                        <tr>
                            <td colspan="5" class="text-center">لا توجد عمليات في الصندوق</td>
                        </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<script>
$(document).ready(function() {
    // Delete fund handler
    $('.delete-fund').click(function(e) {
        e.preventDefault();
        const id = $(this).data('id');
        
        if (confirm('هل أنت متأكد من حذف الصندوق؟')) {
            $.post('index.php?page=cash_funds', {
                action: 'delete_fund',
                fund_id: id
            }).done(function(response) {
                window.location.reload();
            }).fail(function() {
                alert('حدث خطأ أثناء الحذف');
            });
        }
    });

    // Delete transaction handler
    $('.delete-transaction').click(function(e) {
        e.preventDefault();
        const id = $(this).data('id');
        
        if (confirm('هل أنت متأكد من حذف العملية؟')) {
            $.post('index.php?page=cash_funds', {
                action: 'delete_transaction',
                transaction_id: id
            }).done(function(response) {
                window.location.reload();
            }).fail(function() {
                alert('حدث خطأ أثناء الحذف');
            });
        }
    });
});
</script>

<!-- Print Styles -->
<style media="print">
    @page {
        size: A4;
        margin: 1.5cm;
    }
    
    body {
        background: none !important;
    }
    
    .container-fluid {
        width: 100% !important;
        padding: 0 !important;
    }
    
    .card {
        border: none !important;
        box-shadow: none !important;
    }
    
    .card-header {
        display: none !important;
    }
    
    .btn, 
    .modal,
    .alert,
    form,
    .actions-column,
    .btn-close {
        display: none !important;
    }
    
    .table {
        width: 100% !important;
        margin-bottom: 0 !important;
        border-collapse: collapse !important;
    }
    
    .table th,
    .table td {
        padding: 8px !important;
        border: 1px solid #ddd !important;
        text-align: right !important;
    }
    
    .table thead th {
        background-color: #f8f9fa !important;
        border-bottom: 2px solid #ddd !important;
    }
    
    /* Show only date, amount and description columns when printing */
    .table th:not(:nth-child(1)):not(:nth-child(3)):not(:nth-child(4)),
    .table td:not(:nth-child(1)):not(:nth-child(3)):not(:nth-child(4)) {
        display: none !important;
    }
    
    /* Print header styles */
    .print-header {
        text-align: center;
        margin-bottom: 20px;
    }
    
    .print-header img {
        max-width: 100%;
        height: auto;
    }
    
    .print-header h3 {
        margin: 10px 0;
        font-size: 20px;
    }
    
    .print-header p {
        margin: 5px 0;
        font-size: 14px;
    }
    
    .print-date {
        font-size: 12px;
        color: #666;
    }
    
    /* Hide account cards when printing */
    .row.mb-4 {
        display: none !important;
    }
}
</style>

<!-- Add Fund Modal -->
<div class="modal fade" id="addFundModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">إضافة صندوق جديد</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add_fund">
                    <div class="mb-3">
                        <label class="form-label">اسم الصندوق</label>
                        <input type="text" class="form-control" name="name" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">الوصف</label>
                        <textarea class="form-control" name="description" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-primary">حفظ</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Add Deposit Modal -->
<div class="modal fade" id="addDepositModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">إضافة إيداع جديد</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add_transaction">
                    <input type="hidden" name="type" value="deposit">
                    <div class="mb-3">
                        <label class="form-label">الصندوق</label>
                        <select class="form-control" name="fund_id" required>
                            <?php foreach ($cashFunds as $fund): ?>
                            <option value="<?php echo $fund['id']; ?>">
                                <?php echo htmlspecialchars($fund['name']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">المبلغ</label>
                        <input type="number" step="0.01" class="form-control" name="amount" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">رقم المرجع</label>
                        <input type="text" class="form-control" name="reference_number">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">التاريخ</label>
                        <input type="date" class="form-control" name="transaction_date" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">الوصف</label>
                        <textarea class="form-control" name="description" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-success">حفظ</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Add Withdrawal Modal -->
<div class="modal fade" id="addWithdrawalModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">إضافة سحب جديد</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add_transaction">
                    <input type="hidden" name="type" value="withdrawal">
                    <div class="mb-3">
                        <label class="form-label">الصندوق</label>
                        <select class="form-control" name="fund_id" required>
                            <?php foreach ($cashFunds as $fund): ?>
                            <option value="<?php echo $fund['id']; ?>">
                                <?php echo htmlspecialchars($fund['name']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">المبلغ</label>
                        <input type="number" step="0.01" class="form-control" name="amount" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">رقم المرجع</label>
                        <input type="text" class="form-control" name="reference_number">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">التاريخ</label>
                        <input type="date" class="form-control" name="transaction_date" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">الوصف</label>
                        <textarea class="form-control" name="description" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-danger">حفظ</button>
                </div>
            </form>
        </div>
    </div>
</div>
